// Content script for Target orders page
// This runs automatically when on target.com/orders

console.log('Target Budget Sync: Content script loaded');

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'scrapeOrders') {
    const orders = scrapeTargetOrders();
    sendResponse({ orders });
  }
  return true;
});

// Main scraping function
function scrapeTargetOrders() {
  console.log('Target Budget Sync: Starting to scrape orders...');
  const orders = [];

  // Target has multiple possible page structures
  // Try to detect which one we're on

  // Method 1: Order cards with data-test attributes
  let orderElements = document.querySelectorAll('[data-test="orderCard"]');

  // Method 2: Order cards with class names
  if (orderElements.length === 0) {
    orderElements = document.querySelectorAll('[class*="OrderCard"], [class*="order-card"], [class*="OrderHistoryCard"]');
  }

  // Method 3: Try to find order containers by structure
  if (orderElements.length === 0) {
    // Look for elements that contain order links
    const orderLinks = document.querySelectorAll('a[href*="/orders/"]');
    const processedContainers = new Set();

    orderLinks.forEach(link => {
      // Find the parent container that likely holds the full order info
      let container = link.closest('[class*="Card"], [class*="card"], article, section');
      if (container && !processedContainers.has(container)) {
        processedContainers.add(container);
      }
    });

    orderElements = Array.from(processedContainers);
  }

  console.log('Target Budget Sync: Found', orderElements.length, 'potential order elements');

  orderElements.forEach((orderEl, index) => {
    try {
      const order = parseOrderElement(orderEl, index);
      if (order) {
        orders.push(order);
      }
    } catch (e) {
      console.error('Target Budget Sync: Error parsing order element:', e);
    }
  });

  // If still no orders, try a more aggressive approach
  if (orders.length === 0) {
    console.log('Target Budget Sync: Trying alternative parsing method...');
    const altOrders = tryAlternativeParsing();
    orders.push(...altOrders);
  }

  console.log('Target Budget Sync: Successfully scraped', orders.length, 'orders');
  return orders;
}

function parseOrderElement(orderEl, index) {
  // Extract order ID
  let orderId = null;
  const orderLink = orderEl.querySelector('a[href*="/orders/"]');
  if (orderLink) {
    const match = orderLink.href.match(/orders\/(\d+)/);
    if (match) orderId = 'TARGET_' + match[1];
  }

  // Try data attributes
  if (!orderId) {
    orderId = orderEl.dataset.orderId || orderEl.dataset.orderNumber;
    if (orderId) orderId = 'TARGET_' + orderId;
  }

  // Generate fallback ID
  if (!orderId) {
    orderId = 'TARGET_' + Date.now() + '_' + index;
  }

  // Extract order date
  let orderDate = null;
  const dateSelectors = [
    '[data-test="orderDate"]',
    '[class*="orderDate"]',
    '[class*="OrderDate"]',
    '[class*="date"]',
    'time',
    '[datetime]'
  ];

  for (const selector of dateSelectors) {
    const dateEl = orderEl.querySelector(selector);
    if (dateEl) {
      const dateText = dateEl.getAttribute('datetime') || dateEl.textContent;
      const parsed = parseDate(dateText);
      if (parsed) {
        orderDate = parsed;
        break;
      }
    }
  }

  // Extract total amount
  let totalAmount = 0;
  const totalSelectors = [
    '[data-test="orderTotal"]',
    '[class*="orderTotal"]',
    '[class*="OrderTotal"]',
    '[class*="total"]',
    '[class*="Total"]'
  ];

  for (const selector of totalSelectors) {
    const totalEl = orderEl.querySelector(selector);
    if (totalEl) {
      const amount = parsePrice(totalEl.textContent);
      if (amount > 0) {
        totalAmount = amount;
        break;
      }
    }
  }

  // Extract order type
  let orderType = 'online';
  const fulfillmentText = orderEl.textContent.toLowerCase();
  if (fulfillmentText.includes('picked up') || fulfillmentText.includes('in store') || fulfillmentText.includes('in-store')) {
    orderType = 'in_store';
  } else if (fulfillmentText.includes('order pickup') || fulfillmentText.includes('drive up')) {
    orderType = 'pickup';
  } else if (fulfillmentText.includes('shipped') || fulfillmentText.includes('delivered')) {
    orderType = 'shipped';
  }

  // Extract store location if available
  let storeLocation = null;
  const storeEl = orderEl.querySelector('[class*="store"], [class*="Store"], [class*="location"], [class*="Location"]');
  if (storeEl) {
    storeLocation = storeEl.textContent.trim();
  }

  // Extract items
  const items = parseOrderItems(orderEl);

  // Only return order if we have meaningful data
  if (totalAmount > 0 || items.length > 0) {
    return {
      order_id: orderId,
      order_date: orderDate || new Date().toISOString().split('T')[0],
      order_time: null,
      order_type: orderType,
      store_location: storeLocation,
      total_amount: totalAmount,
      payment_method_last4: null,
      items: items
    };
  }

  return null;
}

function parseOrderItems(orderEl) {
  const items = [];

  const itemSelectors = [
    '[data-test="orderItem"]',
    '[class*="orderItem"]',
    '[class*="OrderItem"]',
    '[class*="product"]',
    '[class*="Product"]',
    '[class*="lineItem"]',
    '[class*="LineItem"]'
  ];

  let itemElements = [];
  for (const selector of itemSelectors) {
    itemElements = orderEl.querySelectorAll(selector);
    if (itemElements.length > 0) break;
  }

  itemElements.forEach(itemEl => {
    // Extract item name
    let name = null;
    const nameSelectors = ['h3', 'h4', '[class*="name"]', '[class*="Name"]', '[class*="title"]', '[class*="Title"]', 'a'];
    for (const selector of nameSelectors) {
      const nameEl = itemEl.querySelector(selector);
      if (nameEl && nameEl.textContent.trim().length > 2) {
        name = nameEl.textContent.trim();
        break;
      }
    }

    if (!name) return;

    // Extract price
    let price = 0;
    const priceEl = itemEl.querySelector('[class*="price"], [class*="Price"]');
    if (priceEl) {
      price = parsePrice(priceEl.textContent);
    }

    // Extract quantity
    let quantity = 1;
    const qtyEl = itemEl.querySelector('[class*="qty"], [class*="Qty"], [class*="quantity"], [class*="Quantity"]');
    if (qtyEl) {
      const qtyMatch = qtyEl.textContent.match(/(\d+)/);
      if (qtyMatch) quantity = parseInt(qtyMatch[1]);
    }

    // Try to determine category from item context
    let category = 'Uncategorized';
    const itemText = itemEl.textContent.toLowerCase();
    if (itemText.includes('grocery') || itemText.includes('food') || itemText.includes('beverage')) {
      category = 'Food & Beverage';
    } else if (itemText.includes('household') || itemText.includes('cleaning')) {
      category = 'Household Essentials';
    } else if (itemText.includes('beauty') || itemText.includes('personal care')) {
      category = 'Beauty & Personal Care';
    } else if (itemText.includes('clothing') || itemText.includes('apparel')) {
      category = 'Apparel';
    } else if (itemText.includes('toy') || itemText.includes('game')) {
      category = 'Toys & Games';
    } else if (itemText.includes('electronic')) {
      category = 'Electronics';
    }

    items.push({
      name: name.substring(0, 200), // Limit name length
      quantity: quantity,
      unit_price: price,
      total_price: price * quantity,
      target_category: category
    });
  });

  return items;
}

function tryAlternativeParsing() {
  // This is a fallback method that tries to extract any order-like data from the page
  const orders = [];

  // Look for price patterns and dates near each other
  const pricePattern = /\$\d+\.\d{2}/g;
  const datePattern = /\b(?:Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s+\d{1,2},?\s+\d{4}\b/gi;

  // Get all text nodes with prices
  const bodyText = document.body.innerText;
  const prices = bodyText.match(pricePattern) || [];
  const dates = bodyText.match(datePattern) || [];

  console.log('Target Budget Sync: Alternative parsing found', prices.length, 'prices and', dates.length, 'dates');

  // If we found some structured data, return a placeholder
  if (prices.length > 0 && dates.length > 0) {
    // This indicates there's order data, but we couldn't parse it
    // Return empty to trigger the "no orders found" message which prompts user action
  }

  return orders;
}

function parseDate(dateStr) {
  if (!dateStr) return null;

  // Try to parse various date formats
  const cleaned = dateStr.trim();

  // ISO format
  if (/^\d{4}-\d{2}-\d{2}/.test(cleaned)) {
    return cleaned.substring(0, 10);
  }

  // Try standard Date parsing
  const parsed = new Date(cleaned);
  if (!isNaN(parsed.getTime())) {
    return parsed.toISOString().split('T')[0];
  }

  return null;
}

function parsePrice(priceStr) {
  if (!priceStr) return 0;

  const match = priceStr.match(/\$?([\d,]+\.?\d*)/);
  if (match) {
    return parseFloat(match[1].replace(/,/g, ''));
  }
  return 0;
}

// Expose for testing
window.targetBudgetSync = {
  scrapeOrders: scrapeTargetOrders
};
